#!/bin/sh
# --------------------------------------------------------------------------
# Skript fuer den Import von SQL-Daten in eine LZH-Datenbank.
#
#  1_strassen_Insert.SQL
#  2_objekte_Insert.SQL
#  3_gaknoten_Insert.SQL
#  loesche in umgekehrter Reihenfolge
# (unterstuetzt PostgreSQL oder Oracle).
#
# Autor	        : Energie- & Umweltbuero e.V. (jkr) 
#
# Ausfuehren in     : /home/lzh/bin
# Ausfuehren als    : lzh
#
# Kommandozeilenparameter:
# Objektverzeichnis : z.B. 'GZIE_Lageplan' (4 erste Buchstaben werden als 
#                     Objektname ausgewertet)
# SQL-Quellpfad     : Z.B. '/media/usbplatte/ORAT_AP/sql' 
#
# Datenbank         : pg=Postgres, ora=Oracle, intern fest eingestellt.
#
# Exitcodes         : 0 - Normal beendet.
#                     1 - Allgemeiner Parameterfehler.
#                     2 - Erster Parameter (Objektname) ist kleiner als 4.
#                     3 - Datenbanktyp falsch gesetzt.
#
# Modifiziert   jkr : 2019-07-20 Anpassung nach Muster von import-fenster.sh.
# --------------------------------------------------------------------------

##### Funktionen #####

# Benutzung des Skriptes anzeigen bei falscher Benutzung.
print_usage () {
 echo 
 echo Aufruf: $0 \<Objektverzeichnis\> [\<SQL-Quellpfad\>]
 echo
 echo Beispiel 1: $0 BODD
 echo \-\> Basicdaten\(SQL\) fuer 'BODD' aus Verzeichnis '/home/lzh/gak_import/bodd/sql' importieren.
 echo Hinweis 1: Die ersten vier Zeichen des 1. Parameters \(Objektverzeichnis\) werden zur Bildung des Objektnames 
 echo automatisch in Grossbuchstaben umgewandelt.
 echo /home/lzh/\<Objektverzeichnis\>/sql geloescht.
 echo 
 echo Verlasse Skript mit Exitcode = 1.
 exit 1 					# Exicode 1: Parameter falsch angegeben.
}

# Starte das Skript.
echo
echo Starte Skript \'$0\' mit $# Parameter\(n\).
echo

##### Parametrierung  #####

# Dateien 'profile' und '.bashrc' sourcen (importieren / ausfuehren): 
# Lokale Einstellungen des OS uebernehmen (Umgebungsvariablen u.ae.).
. /etc/profile
. /home/lzh/.bashrc

# Kommandozeilenparameter

# Wenn erster Parameter Laenge 0 (= nicht existiert) hat, abbrechen
if test -z $1; then
 echo Fehler: Paramater falsch angegeben. Bitte beachten Sie die folgenden Hinweise: 
 print_usage;
fi

# Wenn erster Parameter kuerzer als 4 (minimale Laenge Objektname) ist, abbrechen:
OBJEKT_LENGTH=`expr length $1`			# Laenge des ersten Parameters $1 ermitteln.
if [ $OBJEKT_LENGTH -lt 4 ]; then
 echo Fehler: Laenge des ersten Parameters ist kleiner als vier.
 echo Verlasse Skript mit Exitcode = 2.
 exit 2					        # Exitcode 2: Objektname zu kurz.
fi

# Wenn bis hier gekommen, ist Parameter 1 korrekt und kann ausgewertet werden.
OBJEKT_DIR=$1
OBJEKT=`echo ${OBJEKT_DIR:0:4}` 		# Erste vier Buchstaben als Objektname verwenden
echo Objektverzeichnis: $OBJEKT_DIR
echo Objektname: $OBJEKT


# Wenn zweiter Parameter vorhanden ist und ein Verzeichnis, nehme diesen als SQL-Quellpfad (SQL_SOURCE_PATH).
if [[ ( -n $2 ) && ( -d $2 ) ]]; then
 SQL_SOURCE_PATH=$2
 echo SQL-Quellpfad: $SQL_SOURCE_PATH
else
 echo SQL-Quellpfad: Nicht verwendet. 	
fi


# DB-Typ festlegen: Oracle (ora) oder Postgres (pg). Durch Ein- und Auskommentieren einstellen.
#DATABASE_TYPE=ora				# Alternative: DATABASE_TYPE=pg
DATABASE_TYPE=pg				# Alternative: DATABASE_TYPE=ora
echo Datenbanktyp: $DATABASE_TYPE

# Interne Parameter
IMPORT_PATH=/home/lzh/gak_import		# Allgemeiner Pfad, aus der die Daten importiert werden.
echo Allgemeiner Import-Pfad: $IMPORT_PATH
OBJEKT_PATH=$IMPORT_PATH/$OBJEKT_DIR		# Pfad, aus dem aktuelle Datenpunkte importiert werden.
echo Import-Pfad mit Objektverzeichnis: $OBJEKT_PATH
LOG_PATH=$OBJEKT_PATH/_logfiles		        # Pfad zu Logdateien des Objektes.
echo Pfad zu Logdateien: $LOG_PATH		
DATE=`date +%Y-%m-%d_%H%M` 			# Aktuelles Datum als Variable.
echo Aktuelles Datum: $DATE
LOGFILE=$LOG_PATH/$DATE-$OBJEKT-basic.log	# Name des Logfiles fuer Import: Datum mit Objektname und Suffix 'basic'.
echo Logdatei fuer Import: $LOGFILE
SQL_PATH=$OBJEKT_PATH/sql/obj			# Pfad, der die Dateien mit auszufuerenden SQL-Dateien enthaelt.
echo Pfad mit SQL-Dateien: $SQL_PATH

# PostgreSQL-Parameter setzen.
PGDATABASE=lzh					# PG-Datenbankname setzen.
PGUSER=postgres					# PG-Datenbankbenutzer setzen.
PGPASSWORD=lzh.wks				# PG-Datenbankpasswort setzen. vorher:postgres
export PGDATABASE PGUSER PGPASSWORD	   	# Variablen exportieren.

# Oracle-Parameter setzen.
ORAUSER=oeser					# Ora-Datenbankbenutzer setzen.
ORAPASSWORD=bahnhof				# Ora-Datenbankpasswort setzen.
export ORAUSER ORAPASSWORD			# Variablen exportieren.

# PATH anpassen: Dienstprogramme des KP-Prozesses in '/home/lzh/bin' muessen erreichbar sein.
OLD_PATH=$PATH					# Alte Variable PATH sichern.
PATH=.:/home/lzh/bin:$PATH			# Verzeichnis '/home/lzh/bin' und aktuelles Verzeichnis ('.') hinzufuegen.


### Importieren aus Objektverzeichnis.###

# Wenn SQL_SOURCE_PATH leer, kopiere nicht nach SQL_PATH. In diesem Fall muessen die Objektdaten bereits in SQL_PATH liegen.
if [ $SQL_SOURCE_PATH ]; then
 # ACHTUNG: Bereits vorhandene Dateien im Verzeichnis SQL_PATH werden ohne Rueckfrage geloescht!
 echo Loesche vorhandene SQL-Dateien in Verzeichnis $SQL_PATH.
 rm $SQL_PATH/*.sql				# Loesche alle Dateien mit Suffix 'sql' (Kleinschreibung).
 rm $SQL_PATH/*.SQL				# Loesche alle Dateien mit Suffix 'SQL' (Grossschreibung).
 echo Kopiere Daten aus Quellpfad $SQL_SOURCE_PATH nach $SQL_PATH.
 cp -vp $SQL_SOURCE_PATH/* $SQL_PATH  
else
 echo Benutze vorhandene Daten in $SQL_PATH fuer Import.
fi

### SQL-Skripte verarbeiten. ###

echo zeige Verzeichnis: $SQL_PATH
#echo weiter mit ENTER
#read
cd $SQL_PATH
SQL_LIST=`ls *Insert.SQL`

# Jede *Insert.SQL-Datei verarbeiten
for SQL_FILE in $SQL_LIST
do 
# Logdatei schreiben
 echo >> $SQL_FILE
 echo >> $LOGFILE
 echo "-----------------------------------------------" >> $LOGFILE
 echo "Bearbeitung der Datei $SQL_FILE:" >> $LOGFILE
 echo "---> mit TASTE q (quittieren)------------------" >> $LOGFILE
 
 # Prfen, ob fr Postgres oder Oracle importiert wird. 
 if [ "$DATABASE_TYPE" = "pg" ]; then
	# Import der Basicdaten fuer PostgreSQL.
	psql --username=$PGUSER  --dbname=$PGDATABASE -e --file=$SQL_FILE >> $LOGFILE;

 elif [ "$DATABASE_TYPE" = "ora" ]; then
	# Import der Basicdaten fuer Oracle.
	sqlplus $ORAUSER/$ORAPASSWORD \@$SQL_FILE >> $LOGFILE;
 else 
  echo Fehler: Datenbanktyp mit Wert \'$DATABASE_TYPE\' falsch spezifiziert. Umgebungsvariable DATABASE_TYPE muss entweder den Wert \'ora\' oder \'pg\' haben. 
  echo Beende Skript mit Exitcode = 3.
  exit 3					# Exicode 3: DATABASE_TYPE falsch gesetzt.
 fi
 done
echo Skript beenden
### Skript beenden ###

less $LOGFILE				   # Logdatei anzeigen.
echo Raeume Ablaufumgebung auf.				
PATH=$OLD_PATH					# PATH auf alten Wert setzen.
unset PGDATABASE PGUSER PGPASSWORD ORAUSER ORAPASSWORD		   # Gesetzte Variablen wieder entfernen.
echo Verlasse Skript \'$0\' mit Exitcode = 0. Auf Wiedersehen!			
exit 0					   # Verlassen mit Exitcode 0: Alles in Ordnung.
