#!/bin/sh
# --------------------------------------------------------------------------
# Skript fuer den Import einer cfg-Datei in die LZH-Datenbank 
# (unterstuetzt PostgreSQL oder Oracle).
#
# Autor	            : Energie- & Umweltbuero e.V. (tc, mh, kro) 
#
# Ausfuehren in     : /home/lzh/bin
# Ausfuehren als    : lzh
#
# Kommandozeilenparameter:
# Objektverzeichnis : Z.B. 'GZIE_Zeitplan' (4 erste Buchstaben werden als 
#                     Objektname ausgewertet)
# Objektpfad        : Z.B. '/media/usbplatte' 
#log
#
# Datenbank         : pg=Postgres, ora=Oracle, intern fest eingestellt.
#
# Exitcodes         : 0 - Normal beendet.
#                     1 - Allgemeiner Parameterfehler.
#                     2 - Erster Parameter (Objektname) ist kleiner als 4.
#                     3 - Datenbanktyp falsch gesetzt.
#
# Modifiziert   mh  : 2009-09-30 Anpassung fue PostgreSQL-basierte LZHs
# Modifiziert   kro : 2009-11-11
# Modifiziert   mh  : 2009-11-19 Zweiter Parameter ist nun Zusatz fuer 
#                     Objekt-Pfad. 
# Modifiziert   mh  : 2010-01-12 Aenderung der Parameter nach Ruecksprache mit 
#                     Reinhold (siehe Funktion print_usage()).
# --------------------------------------------------------------------------


##### Funktionen #####

# Benutzung des Skriptes anzeigen bei falscher Benutzung.
print_usage () {
 echo 
 echo Aufruf: $0 \<Objektverzeichnis\> [\<Objektpfad\>]
 echo
 echo Beispiel 1: $0 BODD
 echo \-\> Datenpunkte fuer 'BODD' aus Verzeichnis '/home/lzh/gak_import/bodd' importieren.
 echo Beispiel 2: $0 GZIE_Zeitplan 
 echo \-\> Datenpunkte fuer  'GZIE' aus Verzeichnis '/home/lzh/gak_import/GZIE_Zeitplan' importieren.
 echo Beispiel 3: $0 ORAT /media/usbdisk 
 echo \-\> Datenpunkte fuer 'ORAT' aus Verzeichnis '/media/usbdisk/ORAT' importieren.
 echo Beispiel 4: $0 BREF_Anlagenplan /media/disk 
 echo \-\> Datenpunkte fuer  'BREF' aus Verzeichnis '/media/disk/BREF_Anlagenplan' importieren.
 echo
 echo Hinweis 1: Die ersten vier Zeichen des 1. Parameters \(Objektverzeichnis\) werden zur Bildung des Objektnames automatisch in Grossbuchstaben umgewandelt.
 echo Hinweis 2: Der 2. Parameter \(Objektpfad\) muss OHNE abschliessendes \'/\' angegeben werden.
 echo Hinweis 3: Die interne Umgebungsvariable DATABASE_TYPE muss entweder den Wert \'ora\' oder \'pg\' haben.
 echo 
 echo Verlasse Skript mit Exitcode = 1.
 exit 1 					# Exicode 1: Parameter falsch angegeben.
}

# Starte das Skript.
echo
echo Starte Skript \'$0\' mit $# Parameter\(n\).
echo

##### Parametrierung  #####

# Dateien 'profile' und '.bashrc' sourcen (importieren / ausfuehren): Lokale Einstellungen des OS uebernehmen (Umgebungsvariablen u.ae.).
. /etc/profile
. /home/lzh/.bashrc

# Kommandozeilenparameter

# Wenn erster Parameter Laenge 0 (= nicht existiert) hat, abbrechen
if test -z $1; then
 echo Fehler: Paramater falsch angegeben. Bitte beachten Sie die folgenden Hinweise: 
 print_usage;
fi

# Wenn erster Parameter kuerzer als 4 (minimale Laenge Objektname) ist, abbrechen:
OBJEKT_LENGTH=`expr length $1`			# Laenge des ersten Parameters $1 ermitteln.
if [ $OBJEKT_LENGTH -lt 4 ]; then
 echo Fehler: Laenge des ersten Parameters ist kleiner als vier.
 echo Verlasse Skript mit Exitcode = 2.
 exit 2					# Exitcode 2: Objektname zu kurz.
fi

# Wenn bis hier gekommen, ist Parameter 1 korrekt und kann ausgewertet werden.
OBJEKT_DIR=$1
OBJEKT=`echo ${OBJEKT_DIR:0:4}` 		# Erste vier Buchstaben als Objektname verwenden
echo Objektverzeichnis: $OBJEKT_DIR
echo Objektname: $OBJEKT


# Wenn zweiter Parameter vorhanden ist und ein Verzeichnis, nehme diesen als Objektquellpfad (SOURCE_PATH).
if [[ ( -n $2 ) && ( -d $2 ) ]]; then
 SOURCE_PATH=$2
 echo Objekt-Quellpfad: $SOURCE_PATH
else
 echo Objekt-Quellpfad: Nicht verwendet. 	
fi


# DB-Typ festlegen: Oracle (ora) oder Postgres (pg). Durch Ein- und Auskommentieren einstellen.
#DATABASE_TYPE=ora				# Alternative: DATABASE_TYPE=pg
DATABASE_TYPE=pg				# Alternative: DATABASE_TYPE=ora
echo Datenbanktyp: $DATABASE_TYPE

# Interne Parameter
GAK_CFG=gak.cfg					# Name der zu importierenden .cfg-Datei.
echo Zu importierende .cfg-Datei: $GAK_CFG
IMPORT_PATH=/home/lzh/gak_import		# Allgemeiner Pfad, aus der die Daten importiert werden.
echo Allgemeiner Import-Pfad: $IMPORT_PATH
OBJEKT_PATH=$IMPORT_PATH/$OBJEKT_DIR		# Pfad, aus dem aktuelle Datenpunkte importiert werden.
echo Import-Pfad mit Objektverzeichnis: $OBJEKT_PATH
CFG_PATH=$OBJEKT_PATH/cfg			# Pfad zu Konfigurationsdateien des Objektes.
echo Pfad zu Konfigurationsdateien: $CFG_PATH
LOG_PATH=$OBJEKT_PATH/_logfiles		# Pfad zu Logdateien des Objektes.
echo Pfad zu Logdateien: $LOG_PATH	
	
DATE=`date +%Y-%m-%d_%H%M`      	# Aktuelles Datum als Variable.
echo Aktuelles Datum: $DATE
LOGFILE=$LOG_PATH/$DATE-$OBJEKT-datenpunkte.log		# Name des Logfiles fuer Import: Datum mit Objektname und Suffix 'datenpunkte'.
echo Logdatei fuer Import: $LOGFILE
GAK=`echo $OBJEKT | tr [a-z] [A-Z]`_1 		# GAK-Name zu Objekt erzeugen: Grossschreibung des Objektnames mit Suffix '_1'.
echo Name des GA-Knotens: $GAK
ADVISOR=UNBEKANNT				# Betreuer des Objektnames.
echo Betreuer des Objekts: $ADVISOR

# PATH anpassen: Dienstprogramme des KP-Prozesses in '/home/lzh/bin' muessen erreichbar sein.
OLD_PATH=$PATH					# Alte Variable PATH sichern.
PATH=.:/home/lzh/bin:$PATH			# Verzeichnis '/home/lzh/bin' und aktuelles Verzeichnis ('.') hinzufuegen.


### Importieren aus Objektverzeichnisses ###

# Wenn SOURCE_PATH leer, kopiere nicht nach IMPORT_PATH. In diesem Fall muessen die Objektdaten beireits in IMPORT_PATH liegen.
if [ $SOURCE_PATH ]; then
 echo Kopiere Daten aus Quellpfad $SOURCE_PATH/$OBJEKT_DIR nach $OBJEKT_PATH.
 cp -p $SOURCE_PATH/$OBJEKT_DIR $IMPORT_PATH  
else
 echo Benutze vorhandene Daten in $OBJEKT_PATH fuer Import.
fi

### Einspielen von gak.cfg in Datenbank ###

echo Gehe in Verzeichnis $CFG_PATH.
cd $CFG_PATH
echo Kopiere $GAK_CFG nach $DATE-$OBJEKT.cfg
cp -vp $GAK_CFG $DATE-$OBJEKT.cfg		   # Kopieren der gak.cfg-Datei in Datum mit Objektname.
echo Datei $DATE-$OBJEKT.cfg in Unix-Format umwandeln.
#recode /cl $GAK_CFG				# Datei gak.cfg fuer Unix anpassen: '/cl' steht fuer CR-LF (Unix-Zeilenumbruch). Muesste auch per dos2unix gehen.
dos2unix $DATE-$OBJEKT.cfg 			# Zeilenumbrueche fuer Unix anpassen. Vorher recode.
echo Erstelle Iso-Datei $DATE-$OBJEKT-iso.cfg aus Datei $DATE-$OBJEKT.cfg.
oem2iso.exe < $DATE-$OBJEKT.cfg > $DATE-$OBJEKT-iso.cfg # Erzeuge ISO-Datei (Matthias: Warum ist mir nicht klar, uebernommen aus alter Version des Skriptes).

# PostgreSQL-Parameter setzen.
PGDATABASE=lzh					# PG-Datenbankname setzen.
PGUSER=postgres					# PG-Datenbankbenutzer setzen.
PGPASSWORD=lzh.wks				# PG-Datenbankpasswort setzen. vorher:postgres
export PGDATABASE PGUSER PGPASSWORD	   # Variablen exportieren (notwendig fuer cfi_pg.exe von DTF).


### Datenpunkte importieren aus .cfg-Datei ###

# Pruefen, ob fuer Postgres oder Oracle importiert wird. 
# echo "-----------------------------------------------" > $LOGFILE

 if [ "$DATABASE_TYPE" = "pg" ]; then	# Postgres-DB wird benutzt.
 echo Importiere Daten in PostgreSQL-Datenbank. 
 echo "cfg_pg.exe:" # >> $LOGFILE
  echo #>> $LOGFILE
  # .cfg - Datei importiern mit Postgres-Hilfsprogramm von DTF:
  # Aufruf:
  # cfi.exe <Name der CFG-Datei> <Kennung des GA-Knotens> <Kennung des Betreuers> [Optionen] [Testschalter fuer Trace]
  # cfi_pg.exe $DATE-$OBJEKT-iso.cfg $GAK $ADVISOR /i /ti /ta /to /tdcfi_pg.trc >> $LOGFILE
  cfi_pg.exe $DATE-$OBJEKT-iso.cfg $GAK $ADVISOR #>> $LOGFILE
 elif [ "$DATABASE_TYPE" = "ora" ]; then	# Oracle-DB wird benutzt.
  echo Importiere Daten in Oracle-Datenbank.
  echo "gakimport.exe:" #>> $LOGFILE
  echo #>>  $LOGFILE
  # .cfg - Datei importiern mit Oracle-Hilfsprogramm von DTF.
  gakimport.exe -b LZH1 -g $GAK -e $DATE-$OBJEKT-iso.cfg #>> $LOGFILE
 else 
  echo Fehler: Datenbanktyp mit Wert \'$DATABASE_TYPE\' falsch spezifiziert. Umgebungsvariable DATABASE_TYPE muss entweder den Wert \'ora\' oder \'pg\' haben. 
  echo Beende Skript mit Exitcode = 3.
  exit 3					# Exicode 3: DATABASE_TYPE falsch gesetzt.
 fi
 echo "-----------------------------------------------" #>> $LOGFILE


### Skript beenden ###

less $LOGFILE				   # Logdatei anzeigen.
echo Raeume Ablaufumgebung auf.				
PATH=$OLD_PATH					# PATH auf alten Wert setzen.
unset PGDATABASE PGUSER PGPASSWORD		   # Gesetzte Variablen wieder entfernen.
echo Verlasse Skript \'$0\' mit Exitcode = 0. Auf Wiedersehen!			
exit 0					   # Verlassen mit Exitcode 0: Alles in Ordnung.
